;**********************************;
; WASM TSR Support, Keyboard Trap  ;
; By Eric Tauck                    ;
;                                  ;
; Defines:                         ;
;                                  ;
;   Trap16  hook interrupt 16      ;
;   Free16  release interrupt 16   ;
;   Okay16  verify interrupt 16    ;
;                                  ;
; Requires:                        ;
;                                  ;
;   INTR.ASM                       ;
;   SHIFT.ASM                      ;
;**********************************;

; Note: program must define: TSR_SHIFT and TsrKey.

        jmp     _tsr3_end

_tsr_within     DB      0       ;reentrancy flag
_tsr_func       DB      ?       ;primary function
_tsr_func2      DB      ?       ;secondary function

_tsr_int16      LABEL   DWORD   ;original interrupt
                DS      4

;========================================
; Macro to hook the INT 16H hander.

Trap16  MACRO

;--- save old interrupt

        mov     al, 16H                 ;interrupt number
        call    IntGet                  ;get interrupt
        mov     WORD _tsr_int16, bx     ;save segment
        mov     WORD _tsr_int16+2, dx   ;save offset

;--- hook new interrupt

        mov     al, 16H                 ;interrupt number
        mov     dx, ds                  ;segment
        mov     bx, OFFSET _tsr_key     ;offset
        call    IntSet                  ;set interrupt
        ENDM

;========================================
; Macro to unhook the INT 16H handler.
;
; In: seg= TSR segment address in a
;          register or variable.

Free16  MACRO   seg
        push    ds
        mov     ds, seg
        mov     bx, WORD _tsr_int16     ;load offset
        mov     dx, WORD _tsr_int16+2   ;load segment
        pop     ds
        mov     al, 16H                 ;interrupt number
        call    IntSet                  ;restore interrupt
        ENDM

;========================================
; Macro to check that INT 16 can be
; unhooked with Free16.
;
; In: seg= TSR segment address in a
;          register or variable.
;
; Out: ZY= set if okay.

Okay16  MACRO   seg
        push    seg
        mov     al, 16H                 ;interrupt number
        call    IntGet                  ;get interrupt address
        pop     ax
        cmp     bx, OFFSET _tsr_key     ;check offset
        jne     _oky161
        cmp     dx, ax                  ;check segment
_oky161
        ENDM

;========================================
; Check if keystroke is in table.
;
; In: AX= keystroke.
;
; Out: CY= set if found; AX= routine
;      address.
;
; All registers and flags are preserved
; if the key is not found (except CY).

_tsr_check PROC NEAR
        pushf
        push    bx

;--- search table for key match

        mov     bx, OFFSET TsrKey
        jmps    _tsrck2

_tsrck1 seg     cs
        cmp     [bx], ax
        je      _tsrck4
        add     bx, 4
_tsrck2 seg     cs
        cmp     WORD [bx], 0
        jne     _tsrck1

;--- key not found

_tsrck3 pop     bx
        popf
        clc
        ret

;--- check if shift matches

_tsrck4 push    ax
        call    KeyShf
        and     ax, TSR_SHIFT
        cmp     ax, TSR_SHIFT
        pop     ax
        jne     _tsrck3

;--- hot key and shift state matches

        seg     cs
        mov     ax, [bx + 2]
        pop     bx
        popf
        stc
        ret
        ENDP

;========================================
; Hot key handler.
;
; In: AX= address of service routine.

_tsr_hand PROC  NEAR
        pushf
        push    bx
        push    cx
        push    dx
        push    di
        push    si
        push    bp
        push    ds
        push    es
        mov     dx, cs          ;load segment registers
        mov     ds, dx
        mov     es, dx
        inc     _tsr_within     ;block
        call    ax              ;call routine
        dec     _tsr_within     ;unblock
        pop     es
        pop     ds
        pop     bp
        pop     si
        pop     di
        pop     dx
        pop     cx
        pop     bx
        popf
        ret
        ENDP

;========================================
; Interrupt 16H handler.

_tsr_key PROC   FAR
        seg     cs
        cmp     _tsr_within, 0  ;check if blocked
        jne     _tsrky1         ;exit if so

        seg     cs
        mov     _tsr_func, ah   ;save function number

        cmp     ah, 0           ;check if fetch key
        je      _tsrky2
        cmp     ah, 10H         ;check if fetch extended key
        je      _tsrky2
        seg     cs
        mov     _tsr_func2, 1   ;secondary function if needed
        cmp     ah, 1           ;check if keyboard status
        je      _tsrky4
        seg     cs
        mov     _tsr_func2, 10H ;secondary function if needed
        cmp     ah, 11H         ;check if extended keyboard status
        je      _tsrky4

_tsrky1 seg     cs
        jmp     _tsr_int16      ;branch to original keyboard handler

;--- fetch keystroke

_tsrky2 pushf
        seg     cs
        call    _tsr_int16      ;call keyboard handler
        call    _tsr_check      ;check if hot key
        jc      _tsrky3
        iret

;--- hot key activated on fetch keystroke

_tsrky3 call    _tsr_hand       ;call handler
        seg     cs
        mov     ah, _tsr_func   ;reload function
        jmps    _tsrky2

;--- keyboard status

_tsrky4 pushf
        seg     cs
        call    _tsr_int16      ;call keyboard handler
        jz      _tsrky5
        call    _tsr_check      ;check if hot key
        jc      _tsrky6
_tsrky5 retf    2               ;return with flags

;--- hot key activated on status call

_tsrky6 push    ax
        seg     cs
        mov     ah, _tsr_func2  ;retrieve function
        pushf
        seg     cs
        call    _tsr_int16      ;call keyboard handler
        pop     ax
        call    _tsr_hand       ;call handler
        seg     cs
        mov     ah, _tsr_func   ;reload function
        jmps    _tsrky4
        ENDP

_tsr3_end
