/////////////////
// Main Module //
/////////////////

// Future Features:
//
// - Command line load and/or compile
// - Save and restore window location
// - Compile in separate thread -- Compile() i/o will have to be rewritten
//   because fopen doesn't appear to work in a separate thread (probably uses
//   C library variables allocated in the main thread's stack)

#include "Compile.h"
#include "Parser.h"
#include "resource.k"
#include <alloc.h>
#include <epp.h>
#include <direct.h>
#include <stdio.h>

const int BUFSIZE = 0x8000;
const int LINESIZE = 256;
const int BORDER = 2;

const char *STUB = "INSTALL.EXE";
const char *HIST = "Prestall.dat";
const char *HELP = "Prestall.HLP";

const int ID_HISTORY = 20000;
const int MAXHISTORY = 9;
typedef char HISTORY [LINESIZE];

#define EXTEN "pre"
const char *FILTER = "Prestall Files (*." EXTEN ")\000*." EXTEN "\000All Files (*.*)\000*.*\000\000";

const char *SAMPLE =
"; Sample Prestall File\r\n"
"; replace MyApp with your program name\r\n"
"\r\n"
"target\tSETUP.EXE\r\n"
"title\t\"MyApp Install\"\r\n"
"home\t\"c:\\Program Files\\MyApp\"\r\n"
"replace\tMyApp.exe\r\n"
"menu\tMyApp.exe  \"Programs\\MyApp\"";

char HOME [256];
char DATA [256];
char HELPFILE [256];

class MainFrame : public epp_Frame
{
    public:

        MainFrame ();
        ~MainFrame ();

    private:

        void _New ();
        void New ();
        void _Open ();
        void Open (int = -1);
        void _Save ();
        void Save ();
        void SaveAs ();
        void Cut ();
        void Copy ();
        void Paste ();
        void Delete ();
        void _Compile ();
        void Compile ();
        BOOL Check ();
        void About ();
        void Help ();

        void _LoadHistory ();
        void _SaveHistory ();
        int _FindHistory (const char *);
        void _BumpHistory (int);
        void _RemoveHistory (const char *);
        void _InsertHistory (const char *);
        void _RedrawHistory ();

        void SetTitle ();
        void Resize ();

        void OnClose ();
        void OnCommand (int, int, HWND);
        int OnCreate (CREATESTRUCT *);
        void OnDestroy ();
        void OnPaint ();
        void OnSetFocus (HWND);
        void OnSize (int, int, int);

        epp_Menu _Menu;
        epp_Edit _Edit;
        epp_ToolBar _Toolbar;
        epp_Font _Font;

        char _File [256];
        char *_Buffer;
        BOOL _Changed;
        RECT _Rect;

        int _Items;
        HISTORY *_History;

        BOOL _Help;

    DECLARE_HANDLER
};

MainFrame::MainFrame ()
{
    _Buffer = new char [BUFSIZE];
    _History = malloc (LINESIZE * MAXHISTORY);
    _Items = 0;
}

MainFrame::~MainFrame ()
{
    free (_History);
    delete _Buffer;
}

void MainFrame::_New ()
{
    _File [0] = NULL;
    _Edit.SetText ((char *) SAMPLE);
    _Changed = FALSE;
}

void MainFrame::New ()
{
    if (!Check ())
        return;

    _New ();
    SetTitle ();
}

void MainFrame::_Open ()
{
    epp_File file;
    int end;

    if (file.Create (_File, GENERIC_READ, FILE_SHARE_READ, OPEN_EXISTING)) {
        end = file.ReadFile (_Buffer, BUFSIZE);
        if (_Buffer [end - 1] == '\n') end --;
        if (_Buffer [end - 1] == '\r') end --;
        _Buffer [end] = NULL;
        _Edit.SetText (_Buffer);
    } else {
        _RemoveHistory (_File);
        _New ();
        SetTitle ();
        MessageBox ("Could not open file", "Prestall Error", MB_OK | MB_ICONSTOP);
    }
}

void MainFrame::Open (int history)
{
    epp_OpenDialog dialog;

    if (!Check ())
        return;

    if (history != -1) {
        strcpy (_File, _History [history]);
        _BumpHistory (history);
    } else {
        dialog.Create (OFN_HIDEREADONLY | OFN_EXPLORER, 0, "Open File", NULL, NULL, EXTEN, FILTER, 1, *this);
        if (!dialog.OK ()) return;
        strcpy (_File, dialog);
        _InsertHistory (_File);
    }

    SetTitle ();
    _Open ();
}

void MainFrame::_Save ()
{
    epp_File file;
    int line, len;

    if (file.Create (_File, GENERIC_WRITE, 0, CREATE_ALWAYS)) {
        line = 0;
        while (line < _Edit.GetLineCount ()) {
            len = _Edit.LineLength (_Edit.LineIndex (line));
            _Edit.GetLine (line, _Buffer, BUFSIZE);
            _Buffer [len++] = '\r';
            _Buffer [len++] = '\n';
            file.WriteFile (_Buffer, len);
            line ++;
        }
    }

    _Changed = FALSE;
}

void MainFrame::Save ()
{
    if (!*_File)
        SaveAs ();
    else
        _Save ();
}

void MainFrame::SaveAs ()
{
    epp_SaveDialog dialog;

    dialog.Create (OFN_HIDEREADONLY | OFN_EXPLORER, 0, "Save File", NULL, NULL, EXTEN, FILTER, 1, *this);

    if (dialog.OK ()) {
        strcpy (_File, dialog);
        _InsertHistory (_File);
        SetTitle ();
        _Save ();
    }
}

void MainFrame::Cut ()
{
    _Edit.PostMessage (WM_CUT);
}

void MainFrame::Copy ()
{
    _Edit.PostMessage (WM_COPY);
}

void MainFrame::Paste ()
{
    _Edit.PostMessage (WM_PASTE);
}

void MainFrame::Delete ()
{
    _Edit.PostMessage (WM_CLEAR);
}

void MainFrame::_Compile ()
{
    static char drive [MAXDRIVE + MAXDIR];
    char dir [MAXDIR];

    fnsplit (_File, drive, dir, NULL, NULL);
    strcat (drive, dir);
    SetCurrentDirectory (drive);

    wsprintf (drive, "%s%s", HOME, STUB);
    ::Compile (_File, drive);
}

void MainFrame::Compile ()
{
    Save ();

    if (*_File) {
        EnableWindow (FALSE);
        _Compile ();
        EnableWindow (TRUE);
    }
}

BOOL MainFrame::Check ()
{
    char mess [256];

    if (_Changed) {
        wsprintf (mess, "Save changes to %s?", (*_File)? _File : "Untitled");
        switch (MessageBox (mess, "Prestall - Confirm", MB_YESNOCANCEL | MB_ICONQUESTION)) {
            case IDYES:
                Save ();
            case IDNO:
                return TRUE;
            case IDCANCEL:
                return FALSE;
        }
    }
    return TRUE;
}

void MainFrame::About ()
{
    epp_Dialog dialog;

    dialog.DialogBoxParam (IDD_ABOUT, *this);
}

void MainFrame::Help ()
{
    WinHelp (HELPFILE, HELP_FINDER);
    _Help = TRUE;
}

void MainFrame::_LoadHistory ()
{
    FILE *file;

    file = fopen (DATA, "r");

    if (file) {
        while (fgets (_History [_Items], LINESIZE, file)) {
            _History [_Items] [strlen (_History [_Items]) - 1] = NULL;
            _Items ++;
        }
        fclose (file);
    }
}

void MainFrame::_SaveHistory ()
{
    FILE *file;
    int item;

    file = fopen (DATA, "w+");

    if (file) {
        for (item = 0; item < _Items; item ++) {
            fputs (_History [item], file);
            fputs ("\n", file);
        }
        fclose (file);
    }
}

int MainFrame::_FindHistory (const char *history)
{
    int item;

    item = 0;

    while (item < _Items)
        if (!stricmp (_History [item], history))
            return item;
        else
            item ++;

    return -1;
}

void MainFrame::_RemoveHistory (const char *history)
{
    int item;

    item = _FindHistory (history);

    if (item != -1) {
        _Items --;
        while (item < _Items) {
            strcpy (_History [item], _History [item + 1]);
            item ++;
        }
        _RedrawHistory ();
    }
}

void MainFrame::_InsertHistory (const char *history)
{
    int item;

    item = _FindHistory (history);

    if (item != -1)
        _BumpHistory (item);
    else {
        item = _Items;

        if (_Items == MAXHISTORY)
            item --;
        else
            _Items ++;

        while (item) {
            strcpy (_History [item], _History [item - 1]);
            item --;
        }

        strcpy (_History [0], history);
        _RedrawHistory ();
    }
}

void MainFrame::_BumpHistory (int history)
{
    char temp [LINESIZE];

    if (history == 0) return;

    strcpy (temp, _History [history]);

    while (history) {
        strcpy (_History [history], _History [history - 1]);
        history --;
    }

    strcpy (_History [0], temp);
    _RedrawHistory ();
}

void MainFrame::_RedrawHistory ()
{
    epp_Menu menu;
    char line [256];
    int item;

    menu.Attach (_Menu.GetSubMenu (0));

    for (item = 0; item < MAXHISTORY; item ++)
        menu.DeleteMenu (item + ID_HISTORY, MF_BYCOMMAND);

    for (item = 0; item < _Items; item ++) {
        wsprintf (line, "&%u %s", item + 1, _History [item]);
        menu.AppendMenu (MF_STRING, item + ID_HISTORY, line);
    }
}

void MainFrame::SetTitle ()
{
    char title [256];

    wsprintf (title, "Prestall - %s", (*_File)? _File : "Untitled");
    SetWindowText (title);
}

void MainFrame::Resize ()
{
    RECT trect;
    RECT rect;

    _Toolbar.GetClientRect (&trect);
    GetClientRect (&_Rect);
    _Rect.top += trect.bottom + BORDER;
    rect = _Rect;
    rect.left += BORDER;
    rect.right -= BORDER;
    rect.top += BORDER;
    rect.bottom -= BORDER;
    _Edit.MoveWindow (rect);
}

void MainFrame::OnClose ()
{
    if (Check ()) {
        if (_Help)
            WinHelp (HELPFILE, HELP_QUIT);
        Close ();
    }

}

void MainFrame::OnCommand (int notify, int id, HWND)
{
    if ((id >= ID_HISTORY) && (id < ID_HISTORY + MAXHISTORY))
        Open (id - ID_HISTORY);
    else
        switch (id) {
            case IDM_NEW:
                New ();
                break;
            case IDM_OPEN:
                Open ();
                break;
            case IDM_SAVE:
                Save ();
                break;
            case IDM_SAVEAS:
                SaveAs ();
                break;
            case IDM_CUT:
                Cut ();
                break;
            case IDM_COPY:
                Copy ();
                break;
            case IDM_PASTE:
                Paste ();
                break;
            case IDM_DELETE:
                Delete ();
                break;
            case IDM_COMPILE:
                Compile ();
                break;
            case IDM_EXIT:
                PostMessage (WM_CLOSE);
                break;
            case IDM_HELP:
                Help ();
                break;
            case IDM_ABOUT:
                About ();
                break;
            case IDD_EDIT:
                if (notify == EN_CHANGE)
                    _Changed = TRUE;
                break;
        }
}

int MainFrame::OnCreate (CREATESTRUCT *)
{
    RECT rect;

    _Help = FALSE;
    _Font.Create (8, FW_NORMAL, "Helv");

    _Menu.LoadMenu (IDR_MENU);
    SetMenu (_Menu);

    _LoadHistory ();
    _RedrawHistory ();

    rect.left = 16;
    rect.top = 16;
    rect.right = (GetSystemMetrics (SM_CXSCREEN) / 2) + 16;
    rect.bottom = (GetSystemMetrics (SM_CYSCREEN) / 2) + 16;
    MoveWindow (rect);

    const int BUTTONS = 10;
	const int IMAGES [BUTTONS] = { NULL, STD_FILENEW, STD_FILEOPEN, STD_FILESAVE, NULL, STD_CUT, STD_COPY, STD_PASTE, NULL, STD_REDOW };
	const int COMMANDS [BUTTONS] = { NULL, IDM_NEW, IDM_OPEN, IDM_SAVE, NULL, IDM_CUT, IDM_COPY, IDM_PASTE, NULL, IDM_COMPILE };
	_Toolbar.Create (*this, CCS_NOMOVEY, IDB_STD_SMALL_COLOR, IMAGES, COMMANDS, BUTTONS);
	_Toolbar.ShowWindow ();

    rect.left = 0;
    rect.top = 0;
    rect.right = 0;
    rect.bottom = 0;
    _Edit.Create (WS_VISIBLE | WS_HSCROLL | WS_VSCROLL | ES_MULTILINE, rect, *this, IDD_EDIT);
    _Edit.SetFont (_Font);

    Resize ();

    SetFocus (_Edit);

    _New ();
    SetTitle ();

    return 0;
}

void MainFrame::OnPaint ()
{
    epp_PaintContext context;
    context.BeginPaint (*this);
    context.DrawEdge (&_Rect, EDGE_SUNKEN, BF_RECT);
}

void MainFrame::OnDestroy ()
{
    _SaveHistory ();
    _Edit.Close ();
    _Toolbar.Close ();
    _Font.Close ();
}

void MainFrame::OnSetFocus (HWND)
{
    SetFocus (_Edit);
}

void MainFrame::OnSize (int type, int, int)
{
    _Toolbar.SendMessage (WM_SIZE, _wParam, _lParam);
    Resize ();
}

BEGIN_HANDLER (MainFrame, epp_Frame)
    ON_WM_PAINT
    ON_WM_SIZE
    ON_WM_COMMAND
    ON_WM_SETFOCUS
    ON_WM_CLOSE
    ON_WM_CREATE
    ON_WM_DESTROY
END_HANDLER

int epp_Main ()
{
    GetHomeDirectory (HOME);
    wsprintf (DATA, "%s%s", HOME, HIST);
    wsprintf (HELPFILE, "%s%s", HOME, HELP);

    MainFrame frame;
    frame.Create ("Prestall");
    frame.ShowWindow ();
    frame.UpdateWindow ();

    epp_App app;
    app.Run ();

    return 0;
}