// Prestall installer stub program.
//
// Future Features:
//
// - Create log file during install.
// - Install files and links to absolute directories and subdirectories.
// - Onscreen notification that installation isn't finished when running upgrade programs.

#include "trailer.h"
#include "install.k"
#include <epp.h>
#include <stdlib.h>
#include <lzexpand.h>
#include <shlobj.h>

#pragma includelib <lz32.lib>
#pragma includelib <ole32.lib>
#pragma includelib <uuid.lib>
#pragma includelib <shell32.lib>

const unsigned KILOBYTE = 1024;
const unsigned MEGABYTE = 1048576;
const unsigned BUFSIZE = 0x8000;

const int ERROR_NONE =          0;
const int ERROR_TEMPCREATE =    1;
const int ERROR_COPYREAD =      2;
const int ERROR_COPYWRITE =     3;
const int ERROR_EXPANDINIT =    4;
const int ERROR_EXPANDREAD =    5;
const int ERROR_EXPANDWRITE =   6;
const int ERROR_TARGETCREATE =  7;
const int ERROR_EXECUTE =       8;


const unsigned COMMAND_ARGS [COMMAND_CNT] =
{
    0,      // CMD_NONE
    1,      // CMD_EXECUTE
    1,      // CMD_DELETE
    2,      // CMD_INSTALL
    2,      // CMD_REPLACE
    2,      // CMD_COPY
    2,      // CMD_RENAME
    1,      // CMD_MKDIR
    1,      // CMD_RMDIR
    2,      // CMD_MENU
    2,      // CMD_DESK
    1,      // CMD_TARGET
    1,      // CMD_TITLE
    1       // CMD_HOME
};

void SaveShortcut (const TCHAR *name, const TCHAR *program, const TCHAR *directory)
{
    IShellLink *shortcut;
    IPersistFile *file;
    WCHAR wide [MAX_PATH];

    CoInitialize (0);
    if (!CoCreateInstance (CLSID_ShellLink, 0, CLSCTX_INPROC_SERVER, IID_IShellLink, (void **) &shortcut)) {
        shortcut->SetPath (program);
        shortcut->SetWorkingDirectory (directory);
        if (!shortcut->QueryInterface (IID_IPersistFile, (void **) &file)) {
            MultiByteToWideChar (CP_ACP, 0, name, -1, wide, MAX_PATH);
            file->Save (wide, TRUE);
            file->Release ();
        }
        shortcut->Release ();
    }
    CoUninitialize ();
}

void GetSpecialDirectory (HWND hwnd, int dir, TCHAR *path)
{
    ITEMIDLIST *id;

    SHGetSpecialFolderLocation (hwnd, dir, &id);
    SHGetPathFromIDList (id, path);
    IMalloc *ialloc;
    SHGetMalloc (&ialloc);
    ialloc->Free (id);
}

class Data
{
    public:

        Data ();
        ~Data ();

        void Load (epp_File& source, unsigned size);

        void Rewind ();
        void Skip (int = 1);
        BOOL Read (char *);
        BOOL Read (unsigned &);

        const char * Title ();
        const char * Home ();

    private:

        char *_Buffer;
        char *_Title;
        char *_Home;
        char *_Pointer;
        char *_Start;
};

inline void Data::Rewind ()
    { _Pointer = _Start; }

inline const char * Data::Title ()
    { return _Title; };

inline const char * Data::Home ()
    { return _Home; };

const unsigned MAXDATA = 0x8000;
const unsigned MAXFIELD = 0x100;
const unsigned STARTFIELDS = 2;

Data::Data ()
{
    _Buffer = new char [MAXDATA];
    _Title = new char [MAXFIELD];
    _Home = new char [MAXFIELD];
}

Data::~Data ()
{
    delete _Home;
    delete _Title;
    delete _Buffer;
}

void Data::Load (epp_File& source, unsigned size)
{
    source.ReadFile (_Buffer, size);
    _Pointer = _Buffer;
    Read (_Title);
    Read (_Home);
    _Start = _Pointer;
}

void Data::Skip (int count)
{
    while (count --)
        if (*_Pointer) {
            while (*_Pointer)
                _Pointer++;
            _Pointer++;
        }
}

BOOL Data::Read (char *field)
{
    if (*_Pointer) {
        while (*_Pointer)
            *field ++ = *_Pointer++;
        *field ++ = *_Pointer++;
        return TRUE;
    } else {
        *field ++ = NULL;
        return FALSE;
    }
}

BOOL Data::Read (unsigned& number)
{
    char temp [32];

    if (Read (temp)) {
        number = atoi (temp);
        return TRUE;
    } else {
        number = 0;
        return FALSE;
    }
}

// global data
epp_Font BannerFont;
epp_File SourceFile;
Trailer SourceTrailer;
Data SourceData;
epp_File InstallFile;
char InstallName [MAX_PATH];
unsigned ExpandedSize;
char InstallPath [MAX_PATH];

unsigned ExpandedBytes (Data& data)
{
    unsigned size;
    unsigned bytes;
    unsigned command;

    bytes = 0;
    data.Rewind ();

    while (data.Read (command))
        switch (command) {
            case CMD_REPLACE:
            case CMD_INSTALL:
                data.Skip ();
                data.Read (size);
                bytes += size;
                break;
            default:
                data.Skip (COMMAND_ARGS [command]);
                break;
        }

    return bytes;
}

class MainDialog : public epp_Dialog
{
    private:

        int StartInstall ();

        void OnCommand (int, int, HWND);
        BOOL OnInitDialog (HWND, LPARAM);

        int _Bytes;

   DECLARE_HANDLER
};

int MainDialog::StartInstall ()
{
    SECURITY_ATTRIBUTES sa;
    char dir [MAX_PATH];
    char root [4] = "c:\\";
    unsigned long sectors, bytes, clusters, total;
    int ret, end;
    BOOL make;

    root [0] = InstallPath [0];
    if (GetDiskFreeSpace (root, &sectors, &bytes, &clusters, &total)) {
        total = sectors * bytes * clusters;
        if (total < _Bytes)
            ret = MessageBox ("Might not be enough disk space, install anyway?", "Installer", MB_YESNOCANCEL);
        else
            ret = IDYES;
    } else
        ret = MessageBox ("Can't check the disk space, install anyway?", "Installer", MB_YESNOCANCEL);

    if (ret == IDYES) {
        sa.nLength = sizeof (sa);
        sa.lpSecurityDescriptor = NULL;
        sa.bInheritHandle = TRUE;

        make = TRUE;
        end = strlen (InstallPath) - 1;
        if (InstallPath [end] == '\\')
            if ((end == 0) || (InstallPath [end - 1] == ':'))
                make = FALSE;
            else
                InstallPath [end] = NULL;

        if (make) {
            GetCurrentDirectory (sizeof (dir), dir);
            make = !SetCurrentDirectory (InstallPath);
            SetCurrentDirectory (dir);
        }

        if (make && !CreateDirectory (InstallPath, &sa)) {
            MessageBox ("Can't create directory", "Installer", MB_OK);
            ret = IDNO;
        }
    }

    return ret;
}

void MainDialog::OnCommand (int, int id, HWND)
{
    switch (id) {
        case IDOK:
            GetDlgItemText (IDC_PATH, InstallPath, sizeof (InstallPath));
            if (strlen (InstallPath))
                switch (StartInstall ()) {
                    case IDYES:
                        Forward ();
                        break;
                    case IDNO:
                        Close ();
                        break;
                }
            break;
        default:
            Forward ();
            break;
    }
}

BOOL MainDialog::OnInitDialog (HWND, LPARAM)
{
    char temp [512];

    GetModuleFileName (NULL, temp, sizeof (temp));
    SourceFile.Create (temp, GENERIC_READ, FILE_SHARE_READ, OPEN_EXISTING);

    if (SourceFile) {
        SourceFile.SetFilePointer (- sizeof (SourceTrailer), FILE_END);
        SourceFile.ReadFile (&SourceTrailer, sizeof (SourceTrailer));

        if ((SourceTrailer.Head == HEAD) && (SourceTrailer.Tail == TAIL)) {
            SourceFile.SetFilePointer (SourceTrailer.DataOffset);
            SourceData.Load (SourceFile, SourceTrailer.DataSize);
            ExpandedSize = ExpandedBytes (SourceData);
            _Bytes = ExpandedSize + (SourceTrailer.FilesSize);

            epp_Window banner;
            RECT rect;
            banner.Attach (GetDlgItem (IDC_BANNER));
            banner.GetWindowRect (&rect);
            BannerFont.Create (rect.bottom - rect.top, FW_BOLD, TRUE, FALSE, ANSI_CHARSET, "Times New Roman");
            banner.SetFont (BannerFont);

            SetDlgItemText (IDC_BANNER, SourceData.Title ());
            SetWindowText (SourceData.Title ());
            SetDlgItemText (IDC_PATH, SourceData.Home ());

            if (_Bytes > MEGABYTE)
                wsprintf (temp, "%u MB", (_Bytes / MEGABYTE) + 1);
            else
                if (_Bytes > KILOBYTE)
                    wsprintf (temp, "%u KB", (_Bytes / KILOBYTE) + 1);
                else
                    wsprintf (temp, "%u bytes", _Bytes);

            SetDlgItemText (IDC_BYTES, temp);
        } else {
            MessageBox ("Invalid installation file", "Installer", MB_ICONSTOP | MB_OK);
            PostMessage (WM_CLOSE);
        }
    } else {
        MessageBox ("Error opening installation file", "Installer", MB_ICONSTOP | MB_OK);
        PostMessage (WM_CLOSE);
    }

    return TRUE;
}

BEGIN_HANDLER (MainDialog, epp_Dialog)
    ON_WM_COMMAND
    ON_WM_INITDIALOG
END_HANDLER

class InstallDialog : public epp_Dialog
{
    public:

        void Process ();

        BOOL Cancel ();
        int Error ();

    private:

        void Message (const char *message);
        void Progress (unsigned);

        static DWORD pascal InstallThread (void *param);

        BOOL CopyFile (epp_File& target, epp_File& source, unsigned bytes);
        BOOL ExpandFile (epp_File& target, int source, unsigned bytes);
        BOOL SkipFile (int source, unsigned bytes);
        void Expand (unsigned cmd, const char *name, unsigned bytes, int lzhandle);

        void OnClose ();
        void OnCommand (int, int, HWND);
        BOOL OnInitDialog (HWND, LPARAM);

        epp_Progress _Progress;
        char *_Buffer;
        unsigned _TotalBytes;
        unsigned _ByteCount;
        BOOL _Cancel;
        int _Error;

    DECLARE_HANDLER
};

inline BOOL InstallDialog::Cancel ()
    { return _Cancel; }

inline int InstallDialog::Error ()
    { return (_Cancel)? 0 : _Error; }

void InstallDialog::Message (const char *message)
{
    SetDlgItemText (IDC_FILE, message);
}

void InstallDialog::Progress (unsigned bytes)
{
    _ByteCount += bytes;
    _Progress.SetPos ((_ByteCount * 100) / _TotalBytes);
}

BOOL InstallDialog::CopyFile (epp_File& target, epp_File& source, unsigned bytes)
{
    unsigned block;

    while (bytes) {
        if (bytes > BUFSIZE)
            block = BUFSIZE;
        else
            block = bytes;

        if (source.ReadFile (_Buffer, block) < block) {
            _Error = ERROR_COPYREAD;
            return FALSE;
        }
        if (_Cancel)
            return FALSE;
        Progress (block);
        if (target.WriteFile (_Buffer, block) < block) {
            _Error = ERROR_COPYWRITE;
            return FALSE;
        }
        if (_Cancel)
            return FALSE;
        Progress (block);
        bytes -= block;
    }
    return TRUE;
}

BOOL InstallDialog::ExpandFile (epp_File& target, int source, unsigned bytes)
{
    unsigned block;

    while (bytes) {
        if (bytes > BUFSIZE)
            block = BUFSIZE;
        else
            block = bytes;

        if (LZRead (source, _Buffer, block) < block) {
            _Error = ERROR_EXPANDREAD;
            return FALSE;
        }
        if (_Cancel)
            return FALSE;
        Progress (block);
        if (target.WriteFile (_Buffer, block) < block) {
            _Error = ERROR_EXPANDWRITE;
            return FALSE;
        }
        if (_Cancel)
            return FALSE;
        Progress (block);
        bytes -= block;
    }
    return TRUE;
}

BOOL InstallDialog::SkipFile (int source, unsigned bytes)
{
    unsigned block;

    while (bytes) {
        if (bytes > BUFSIZE)
            block = BUFSIZE;
        else
            block = bytes;

        if (LZSeek (source, block, 1) < 0) {
            _Error = ERROR_EXPANDREAD;
            return FALSE;
        }
        if (_Cancel)
            return FALSE;

        Progress (block);
        Progress (block);
        bytes -= block;
    }
    return TRUE;
}

void InstallDialog::Expand (unsigned cmd, const char *full, unsigned bytes, int lzhandle)
{
    epp_File target;
    char mess [256];

    if ((cmd == CMD_INSTALL) && target.Create (full, 0, FILE_SHARE_READ | FILE_SHARE_READ, OPEN_EXISTING)) {
        wsprintf (mess, "Skipping %s", full);
        Message (mess);
        if (!SkipFile (lzhandle, bytes)) {
            LZClose (lzhandle);
            return;
        }
    } else {
        wsprintf (mess, "Expanding %s", full);
        Message (mess);
        if (!target.Create (full, GENERIC_WRITE, 0, CREATE_ALWAYS)) {
            LZClose (lzhandle);
            _Error = ERROR_TARGETCREATE;
            return;
        }
        if (!ExpandFile (target, lzhandle, bytes)) {
            LZClose (lzhandle);
            return;
        }
    }
}

static void Shortcut (HWND win, int dir)
{
    char file [256];
    char sdir [256];
    char name [256];
    char full [256];

    SourceData.Read (name);
    wsprintf (file, "%s%s", InstallPath, name);
    GetSpecialDirectory (win, dir, sdir);
    SourceData.Read (name);
    wsprintf (full, "%s\\%s.lnk", sdir, name);
    SaveShortcut (full, file, InstallPath);
}

void InstallDialog::Process ()
{
    epp_Process process;
    char full [MAX_PATH], full2 [MAX_PATH];
    char name [256];
    char mess [256];
    unsigned command;
    unsigned bytes;
    int lzhandle;

    Message ("Copying installation data");
    GetTempFileName (InstallPath, "INS", 0, InstallName);
    if (!InstallFile.Create (InstallName, GENERIC_READ | GENERIC_WRITE, 0, CREATE_ALWAYS)) {
        _Error = ERROR_TEMPCREATE;
        return;
    }
    SourceFile.SetFilePointer (SourceTrailer.FilesOffset);
    if (!CopyFile (InstallFile, SourceFile, SourceTrailer.FilesSize))
        return;

    InstallFile.SetFilePointer (0);
    lzhandle = LZInit ((int) (HANDLE) InstallFile);
    if (lzhandle < 0) {
        _Error = ERROR_EXPANDINIT;
        return;
    }

    SourceData.Rewind ();
    SetCurrentDirectory (InstallPath);

    while (SourceData.Read (command))
        switch (command) {
            case CMD_INSTALL:
            case CMD_REPLACE:
                SourceData.Read (name);
                SourceData.Read (bytes);
                wsprintf (full, "%s%s", InstallPath, name);
                Expand (command, full, bytes, lzhandle);
                break;
            case CMD_EXECUTE:
                SourceData.Read (name);
                wsprintf (mess, "Executing %s", name);
                Message (mess);
                if (process.Create (name))
                    WaitForSingleObject (process, INFINITE);
                else {
                    _Error = ERROR_EXECUTE;
                    return;
                }
                break;
            case CMD_DELETE:
                SourceData.Read (name);
                wsprintf (full, "%s%s", InstallPath, name);
                wsprintf (mess, "Deleting %s", full);
                Message (mess);
                DeleteFile (full);
                break;
            case CMD_RENAME:
                SourceData.Read (name);
                wsprintf (full, "%s%s", InstallPath, name);
                SourceData.Read (name);
                wsprintf (full2, "%s%s", InstallPath, name);
                wsprintf (mess, "Renaming %s", full);
                Message (mess);
                MoveFile (full, full2);
                break;
            case CMD_MENU:
                Message ("Adding to start menu");
                Shortcut (*this, CSIDL_STARTMENU);
                break;
            case CMD_DESK:
                Message ("Adding to desktop");
                Shortcut (*this, CSIDL_DESKTOP);
                break;
            default:
                Message ("Skipping script command");
                SourceData.Skip (COMMAND_ARGS [command]);
                break;
        }
    LZClose (lzhandle);
}

static DWORD pascal InstallDialog::InstallThread (void *param)
{
    ((InstallDialog *) param)->Process ();
    ((InstallDialog *) param)->PostMessage (WM_CLOSE);

    if (InstallFile) {
        InstallFile.Close ();
        DeleteFile (InstallName);
    }

    return 0;
}

void InstallDialog::OnClose ()
{
    Close ();
}

void InstallDialog::OnCommand (int, int id, HWND)
{
    switch (id) {
        case IDC_CANCEL:
            _Cancel = TRUE;
            break;
    }
}

BOOL InstallDialog::OnInitDialog (HWND, LPARAM)
{
    epp_Window banner;

    banner.Attach (GetDlgItem (IDC_BANNER));
    banner.SetFont (BannerFont);

    SetDlgItemText (IDC_BANNER, SourceData.Title ());
    SetWindowText (SourceData.Title ());

    _Progress.Attach (GetDlgItem (IDC_PROGRESS));
    _Progress.SetRange (0, 100);

    _Buffer = new char [BUFSIZE];
    _TotalBytes = (SourceTrailer.FilesSize * 2) + (ExpandedSize * 2);   // read LZ, write LZ, ...
    _ByteCount = 0;                                                     //   read to expand, write file
    _Cancel = FALSE;
    _Error = ERROR_NONE;

    epp_Thread thread;
    thread.Create (InstallThread, this);

    return TRUE;
}

BEGIN_HANDLER (InstallDialog, epp_Dialog)
    ON_WM_COMMAND
    ON_WM_CLOSE
    ON_WM_INITDIALOG
END_HANDLER

int epp_Main ()
{
    char mess [256];
    int len, flag;

    MainDialog *mdlg = new MainDialog;
    InstallDialog *idlg = new InstallDialog;

    mdlg->DialogBoxParam (IDD_MAIN, HWND_DESKTOP);
    if (mdlg->OK ()) {
        len = strlen (InstallPath);
        if (len && (InstallPath [len - 1] != '\\'))
            strcpy (InstallPath + len, "\\");
        idlg->DialogBoxParam (IDD_INSTALL, HWND_DESKTOP);

        if (idlg->Cancel ()) {
            strcpy (mess, "Installation cancelled");
            flag = MB_ICONSTOP;
        } else
            if (idlg->Error ()) {
                wsprintf (mess, "Installation error %i", idlg->Error ());
                flag = MB_ICONSTOP;
            } else {
                strcpy (mess, "Installation successful!");
                flag = MB_ICONINFORMATION;
            }

        MessageBox (NULL, mess, "Installer", MB_OK | MB_SETFOREGROUND | flag);
    }
    delete mdlg, idlg;
    return 0;
}