// Dexter Upgrade Program

#include <epp.h>
#include <fstream.h>
#include <stdlib.h>

// need just for WIDTHX
#include "global.h"

const int MAXCARDS = 8000;              // maximum cards in old and new libraries
const int MAXDECKCARDS = 50000;         // maximum cards in all old decks

struct CARD
{
    int     Key;
    char    Name [50];
};

struct DECK
{
    RECT    Rect;
    char    Name [50];
    BOOL    Locked;
    int     Sort;
};

struct DECKCARD
{
    int     Count;
    char    Name [50];
};

struct HEADER
{
    RECT    Rect;
};

#define MissingTitle "Cards Not Upgraded"       // make a macro to insert in message below

// messages

const char Mess1 [] =
    "I don't know how to upgrade this Dexter!  You might be upgrading an older version of Dexter over a "
    "newer version.  Your existing cards and decks have been saved in DEXTER.BK1 and DEXTER.BK2.  To save "
    "these cards and decks: reinstall your previous version of Dexter to a different directory and copy "
    "DEXTER.BK1 to DEXTER.CRD and DEXTER.BK2 to DEXTER.DAT in that directory.";

const char Mess2 [] =
    "There were cards in your previous version of Dexter that are not in this version of Dexter.  You "
    "either added custom cards or some card names have been changed (probably corrected).  The names of "
    "these cards have been transferred to this version of Dexter and put in a deck called '"
    MissingTitle "'.";

// file names

const char CARDFILE_OLD[] = "DEXTER.BK1";
const char CARDFILE_NEW[] = "DEXTER.CRD";
const char DECKFILE_OLD[] = "DEXTER.BK2";
const char DECKFILE_NEW[] = "DEXTER.DAT";

// version constants

const int SIGNATURE1 = 0x44580001;
const int SIGNATURE2 = 0x44580002;
const int SAVED_END1 = 0;
const int SAVED_CARD1 = 1;
const int SAVED_DECK1 = 2;
const int SORT_NAME1 = 0;
const int SORT_TYPE1 = 1;
const char DELIM1 = '\xFE';

///////////////
// Basic I/O //
///////////////

void SkipToDelim (istream& stream)
{
	char ch;

	stream.unsetf (ios::skipws);
	do
		stream >> ch;
	while (stream && (ch != DELIM1));
	stream.setf (ios::skipws);
}

void ReadToDelim (istream& stream, char *text)
{
	char ch;

	stream.unsetf (ios::skipws);
	stream >> ch;
	while (stream && (ch != DELIM1)) {
		*text++ = ch;
		stream >> ch;
	}
	*text = NULL;
	stream.setf (ios::skipws);
}

void ReadRect (istream& stream, RECT& rect)
{
	stream >> rect.left;
	stream >> rect.top;
	stream >> rect.right;
	stream >> rect.bottom;
}

void WriteRect (ostream& stream, const RECT& rect)
{
	stream << rect.left << ' ';
	stream << rect.top << ' ';
	stream << rect.right << ' ';
	stream << rect.bottom << ' ';
}

/////////////
// Library //
/////////////

class Library
{
    public:

        Library ();
        ~Library ();

        int Load (const char *name);
        const DECKCARD * Card (int key);
        int Key (const DECKCARD *card);

        void Rewind ();
        int Extra ();
        BOOL Dump (int& key);

    private:

        void Save ();

        void LoadCard (istream& stream, CARD& card);
        void SaveCard (ostream& stream, CARD& card);

        epp_Array <CARD> _Cards;
        char _Name [256];
        DECKCARD _Card;
        int _Signature;
        int _Count;
        int _Extra;
        int _Dump;
};

Library::Library ()
{
    _Cards.Alloc (MAXCARDS);
    _Count = 0;
    _Extra = 0;
}

Library::~Library ()
{
    if (_Extra) Save ();
}

int Library::Load (const char *name)
{
    fstream file;

    strcpy (_Name, name);

    file.open (_Name, ios::in);
    file >> _Signature;

    if (!file)
        return 0;

    if ((_Signature != SIGNATURE1) && (_Signature != SIGNATURE2))
        return -1;

    while (file) {
        LoadCard (file, _Cards [_Count]);
        _Count ++;
    }

    return _Count;
}

void Library::Save ()
{
    fstream file;
    int idx;

    file.open (_Name, ios::in | ios::out | ios::app | ios::nocreate);
    idx = _Count - _Extra;

    while (_Extra) {
        SaveCard (file, _Cards [idx]);
        idx ++;
        _Extra --;
    }
}

const DECKCARD * Library::Card (int key)
{
    int i;

    if (key == 0)
        return NULL;

    switch (_Signature) {
        case SIGNATURE1:
        case SIGNATURE2:
            _Card.Count = ((key & 0xFFFF0000) >> 16) + 1;
            key = key & 0x0000FFFF;
    }

    i = 0;
    while (i < _Count)
        if (_Cards [i].Key == key) {
            strcpy (_Card.Name, _Cards [i].Name);
            return &_Card;
        } else
            i ++;

    strcpy (_Card.Name, "Invalid Card");
    return &_Card;
}

int Library::Key (const DECKCARD *card)
{
    int key;
    int i;

    if (card == NULL)
        return 0;

    i = 0;

    while (i < _Count)
        if (!stricmp (card->Name, _Cards [i].Name))
            return _Cards [i].Key | ((card->Count - 1) << 16);
        else
            i ++;

    key = 50000 + _Extra;
    _Cards [_Count].Key = key;
    strcpy (_Cards [_Count].Name, card->Name);

    _Count ++;
    _Extra ++;

    return key | ((card->Count - 1) << 16);
}

void Library::Rewind ()
{
    _Dump = 0;
}

int Library::Extra ()
{
    _Dump = _Count - _Extra;
    return _Extra;
}

BOOL Library::Dump (int& key)
{
    if (_Dump < _Count) {
        key = _Cards [_Dump++].Key;
        return TRUE;
    } else
        return FALSE;
}

void Library::LoadCard (istream& stream, CARD& card)
{
    switch (_Signature) {
        case SIGNATURE1:
        case SIGNATURE2:
        	stream >> card.Key;
        	SkipToDelim (stream);
        	ReadToDelim (stream, card.Name);
        	SkipToDelim (stream);
        	SkipToDelim (stream);
        	SkipToDelim (stream);
        	SkipToDelim (stream);
        	SkipToDelim (stream);
        	SkipToDelim (stream);
        	SkipToDelim (stream);
        	break;
    }
}

void Library::SaveCard (ostream& stream, CARD& card)
{
    switch (_Signature) {
        case SIGNATURE1:
        case SIGNATURE2:
        	stream << card.Key << ' ' << DELIM1;
        	stream << card.Name << DELIM1;
        	stream << DELIM1;
        	stream << DELIM1;
        	stream << DELIM1;
        	stream << DELIM1;
        	stream << DELIM1;
        	stream << DELIM1;
        	stream << DELIM1;
        	stream << endl;
        break;
    }
}

//////////
// Deck //
//////////

class Deck
{
    public:

        BOOL Load (const char *name);
        void Create (const char *name);
        void Rewind ();
        void Close ();

        void operator>> (HEADER& head);
        void operator<< (HEADER& head);
        BOOL operator>> (DECK& deck);
        void operator<< (DECK& deck);
        BOOL operator>> (int&);
        void operator<< (int);

    private:

        BOOL _Nice;
        int _Signature;
        int _Start;
        fstream _File;
};

BOOL Deck::Load (const char *name)
{
    char nice [256];

    _File.open (name, ios::in);

    _File.getline (nice, sizeof (nice));
    if (!stricmp (nice, "NICE")) {
        _Nice = TRUE;
        _File >> _Signature;
    } else {
        _Nice = FALSE;
        _Signature = atoi (nice);
    }


    if (_File) {
        _Start = _File.tellg ();
        return (_Signature == SIGNATURE1) || (_Signature == SIGNATURE2);
    } else
        return FALSE;
}

void Deck::Create (const char *name)
{
    _Nice = FALSE;
    _File.open (name, ios::in | ios::out | ios::trunc);
}

void Deck::Rewind ()
{
    _File.clear ();
    _File.seekg (_Start);
}

void Deck::Close ()
{
    _File.close ();
}

void Deck::operator>> (HEADER& head)
{
    switch (_Signature) {
        case SIGNATURE1:
        case SIGNATURE2:
            ReadRect (_File, head.Rect);
            break;
    }
}

void Deck::operator<< (HEADER& head)
{
    if (_Nice)
        _File << "NICE" << endl;

    _File << SIGNATURE2 << endl;
    WriteRect (_File, head.Rect);
    _File << endl;
}

BOOL Deck::operator>> (DECK& deck)
{
    int type;

    switch (_Signature) {
        case SIGNATURE1:
        case SIGNATURE2:
            _File >> type;
            if (!_File || (type == SAVED_END1))
                return FALSE;
            SkipToDelim (_File);
            ReadToDelim (_File, deck.Name);
            ReadRect (_File, deck.Rect);
            _File >> deck.Locked;
            _File >> deck.Sort;
            break;
    }
    return TRUE;
}

void Deck::operator<< (DECK& deck)
{
    _File << SAVED_DECK1 << ' ' << DELIM1 << deck.Name << DELIM1 << ' ';
    WriteRect (_File, deck.Rect);
    _File << (int) deck.Locked << ' ' << deck.Sort;
    _File << endl;
}

BOOL Deck::operator>> (int& key)
{
    _File >> key;
    return (_File)? TRUE : FALSE;
}

void Deck::operator<< (int key)
{
    _File << key << endl;
}

//////////////////
// Main Program //
//////////////////

int epp_Main ()
{
    epp_Array <int> cardlist (MAXDECKCARDS);
    int cardcount;
    Library oldlib;
    Library newlib;
    Deck olddeck;
    Deck newdeck;
    HEADER head;
    DECK deck, deck2;
    int card;
    int extra;
    int skip;
    int loaded;
    int key;

    loaded = oldlib.Load (CARDFILE_OLD);
    if (loaded > 0)
        if (!olddeck.Load (DECKFILE_OLD))
            loaded = -1;

    newlib.Load (CARDFILE_NEW);
    newdeck.Create (DECKFILE_NEW);

    if (loaded > 0) {
        // save converted deck cards into array

        cardcount = 0;
        olddeck >> head;
        while (olddeck >> deck) { }
        while (olddeck >> key)
            cardlist [cardcount ++] = newlib.Key (oldlib.Card (key));
        olddeck.Rewind ();

        // deck header
        olddeck >> head;
        newdeck << head;

        // copy decks

        extra = newlib.Extra ();
        skip = 0;

        while (olddeck >> deck) {
            if (extra)
                if (stricmp (deck.Name, MissingTitle) >= 0) {
                    deck2.Rect.left = 0;
                    deck2.Rect.top = 0;
                    deck2.Rect.bottom = deck2.Rect.top + (GetSystemMetrics (SM_CYSCREEN) / 2);
                    deck2.Rect.right = deck2.Rect.left + WIDTHX + (GetSystemMetrics (SM_CXFRAME) * 2) + (GetSystemMetrics (SM_CXDLGFRAME) * 2) + GetSystemMetrics (SM_CXVSCROLL);
                    strcpy (deck2.Name, MissingTitle);
                    deck2.Locked = TRUE;
                    deck2.Sort = SORT_NAME1;
                    newdeck << deck2;
                    extra = 0;
                } else
                    skip ++;
            newdeck << deck;
        }
        newdeck << SAVED_END1;

        // card lists


        extra = newlib.Extra ();
        card = 0;

        while ((card < cardcount) && (!extra || skip > 0)) {
            key = cardlist [card ++];
            newdeck << key;
            if (!key) skip --;
        }

        if (extra) {
            while (newlib.Dump (key))
                newdeck << key;
            newdeck << (int) 0;
        }

        while (card < cardcount)
            newdeck << cardlist [card ++];

        if (newlib.Extra ())
            MessageBox (NULL, Mess2, "Dexter Upgrade", MB_OK | MB_ICONINFORMATION);

        olddeck.Close ();

        if (newlib.Extra () < 10) {
            DeleteFile (CARDFILE_OLD);
            DeleteFile (DECKFILE_OLD);
        }
    } else {
        head.Rect.left = 0;
        head.Rect.top = 0;
        head.Rect.right = GetSystemMetrics (SM_CXSCREEN) / 2;
        head.Rect.bottom = GetSystemMetrics (SM_CYSCREEN) / 2;
        OffsetRect (&head.Rect, head.Rect.right / 2, head.Rect.bottom / 2);
        newdeck << head;

        deck.Rect.left = 0;
        deck.Rect.top = 0;
        deck.Rect.bottom = deck.Rect.top + (GetSystemMetrics (SM_CYSCREEN) / 2);
        deck.Rect.right = deck.Rect.left + WIDTHX + (GetSystemMetrics (SM_CXFRAME) * 2) + (GetSystemMetrics (SM_CXDLGFRAME) * 2) + GetSystemMetrics (SM_CXVSCROLL);
        strcpy (deck.Name, "All Cards");
        deck.Locked = TRUE;
        deck.Sort = SORT_NAME1;
        newdeck << deck;
        newdeck << SAVED_END1;

        newlib.Rewind ();
        while (newlib.Dump (key))
            newdeck << key;
        newdeck << (int) 0;

        if (loaded < 0)
            MessageBox (NULL, Mess1, "Dexter Upgrade", MB_OK | MB_ICONSTOP);
    }

    return 0;
}