////////////////////
// Master Library //
////////////////////

#include "Dialogs.h"
#include "Library.h"
#include "Filter.h"
#include "Basic IO.h"
#include <fstream.h>
#include <string.h>
#include <stdio.h>

LibraryClass Library;

static void SetAbilities (CARD_INFO& card);
static istream& operator>> (istream& stream, CARD_INFO& card);
static ostream& operator<< (ostream& stream, CARD_INFO& card);

static const int MAXLIBRARY = 5000;

LibraryClass::LibraryClass ()
{
    _Loaded = FALSE;
}

void LibraryClass::Load ()
{
	fstream file;
	char name [256];
	CARD_INFO card;

	_Data.Alloc (MAXLIBRARY);
	_Cards = 0;

	LoadString (IDS_CARDFILE, name);
	file.open (name, ios::in);

	int signature;
	file >> signature;

	while (TRUE)
		if (file >> card) {
			_Data [_Cards ++] = (card);
		} else
			break;

    _Flush = TRUE;
    _Changed = FALSE;
    _Loaded = TRUE;
}

void LibraryClass::Save ()
{
	fstream file;
	char name [256];
	CARD_INFO card;

	if (_Changed) {
    	LoadString (IDS_CARDFILE, name);
    	file.open (name, ios::in | ios::out | ios::trunc);
    	file << SIGNATURE << endl;
    	for (int index = 0; index < _Cards; index ++)
            file << _Data [index];
    }
}

void LibraryClass::Import (epp_ListBox& newlist, epp_ListBox& updatelist, const char *name)
{
    DevjoeFilter filter;
	fstream file;
	CARD_INFO card;
	int index;

	file.open (name, ios::in);

	while (filter.Read (file, card)) {
        index = _Find (card.Name);
        if (index < 0) {
            New (card);
            newlist.AddItemData (card.Key);
        } else {
            card.Key = _Data [index].Key;
            Update (card);
            updatelist.AddItemData (card.Key);
        }
    }
}

int LibraryClass::New (CARD_INFO& card)
{
    int index;
    BOOL slot;

    slot = FALSE;
    index = 0;

    while (!slot && index < _Cards)
        if (_Data [index].Deleted)
            slot = TRUE;
        else
            index ++;

    if (slot)
        card.Key = _Data [index].Key;
    else {
        index = _Cards;
        _Cards ++;

        if (index)
            card.Key = _Data [index - 1].Key + 1;
        else
            card.Key = 1;
    }

    SetAbilities (card);

    _Data [index] = card;
    _Flush = TRUE;
    _Changed = TRUE;

    return card.Key;
}

inline void HALVE (int& number, int& fraction)
{
	fraction = number & 1;
	number = number / 2;
}

int LibraryClass::Find (const char *name)
{
    int index;

    index = _Find (name);

    if (index < 0)
        return 0;
    else
        return _Data [index].Key;
}

void LibraryClass::Find (int key, CARD_INFO& card)
{
    static CARD_INFO card1;
    static CARD_INFO card2;
    static int key1;
    static int key2;
    static int toggle;

    key &= KEYMASK;

    if (_Flush) {
        key1 = 0;
        key2 = 0;
        toggle = 0;
        _Flush = FALSE;
    }

    if (key1 == key)
        card = card1;
    else if (key2 == key)
        card = card2;
    else {
    	card = _Data [_Find (key)];
        if (toggle) {
            key2 = key;
            card2 = card;
            toggle = 0;
        } else {
            key1 = key;
            card1 = card;
            toggle = 1;
        }
    }
}

void LibraryClass::Update (CARD_INFO& card)
{
    SetAbilities (card);
	_Data [_Find (card.Key)] = card;
    _Flush = TRUE;
    _Changed = TRUE;
}

void LibraryClass::Delete (int key)
{
    CARD_INFO card;

    memset (&card, 0, sizeof (card));
    card.Key = key;
    card.Deleted = TRUE;

    _Data [_Find (key & KEYMASK)] = card;
    _Flush = TRUE;
    _Changed = TRUE;
}

int LibraryClass::_Find (const char *name)
{
    int index;
    int len;

    len = strlen (name) + 1;
    index = 0;

    while (index < _Cards)
        if (!_Data [index].Deleted && !memicmp (_Data [index].Name, name, len))
            return index;
        else
            index ++;

    return -1;
}

int LibraryClass::_Find (int key)
{
	int fraction;
	int fraction2;
	int index;
	int half;

	index = _Cards - 1;
	half = index;
	HALVE (half, fraction);
	index -= half;

	while (key != _Data [index].Key) {
		fraction2 = fraction;
		HALVE (half, fraction);
		if (key < _Data [index].Key)
			index -= half + fraction2;
		else
			index += half + fraction2;
	}

    return index;
}

static BOOL MatchString (const char *field, const char *string, BOOL anywhere, BOOL not)
{
    BOOL instring;

    if (*string) {
        if (anywhere)
            instring = (strustr (field, string))? TRUE : FALSE;
        else
            instring = (memicmp (field, string, strlen (string)))? FALSE : TRUE;
        return (not)? !instring : instring;
    } else
        return FALSE;
}

static BOOL MatchCard (CARD_INFO& card, const CARD_SEARCH& search)
{
    BOOL stringmatch;
    BOOL flagmatch;
    BOOL flagmatch2;

    stringmatch =
        (!*search.Name && !*search.Type && !*search.Spec && !*search.Desc) ||
        MatchString (card.Name, search.Name, search.NameAnywhere, search.SkipMatch) ||
        MatchString (card.Type, search.Type, search.TypeAnywhere, search.SkipMatch) ||
        MatchString (card.Spec, search.Spec, TRUE, search.SkipMatch) ||
        MatchString (card.Desc, search.Desc, TRUE, search.SkipMatch);

    flagmatch = ((card.Abilities & search.ClearFlags) == 0) &&
        ((card.Abilities & search.SetFlags) == search.SetFlags);

    flagmatch2 = ((card.Edition & search.ClearEdition) == 0) &&
        ((card.Edition & search.SetEdition) == search.SetEdition);

    return stringmatch && flagmatch && flagmatch2;
}

void LibraryClass::FindCards (epp_ListBox& listbox, const CARD_SEARCH *search)
{
    int index;

    for (index = 0; index < _Cards; index ++)
        if (!_Data [index].Deleted && (!search || MatchCard (_Data [index], *search)))
            listbox.AddItemData (_Data [index].Key);
}

void LibraryClass::FindCards (epp_ListBox& source, epp_ListBox& listbox, const CARD_SEARCH *search)
{
    CARD_INFO data;
    int key;
    int total;
    int index;

    total = source.GetCount ();

    for (index = 0; index < total; index ++) {
        key = source.GetItemData (index);
        Find (key, data);
        if (!search || MatchCard (data, *search))
            listbox.AddItemData (key);
    }
}

// functions

static void GetFreq (char *freq, char *& ptr)
{
    char c;

    c = *freq++ = *ptr ++;

    if (c != '-')
        while (*ptr && (*ptr >= '0') && (*ptr <='9'))
            *freq++ = *ptr ++;

    *freq = NULL;
}

static void SetAbilities (CARD_INFO& card)
{
    card.Deleted = (strlen (card.Name))? FALSE : TRUE;

	strupr (card.Cast);

	card.Abilities = 0;

	if (strustr (card.Spec, "PHASING")) card.Abilities |= PHASING;
	if (strustr (card.Spec, "PROTECT")) card.Abilities |= PROTECTION;
	if (strustr (card.Spec, "REGENERATE")) card.Abilities |= REGENERATE;
	if (strustr (card.Spec, "FLY")) card.Abilities |= FLYING;
	if (strustr (card.Spec, "WALK")) card.Abilities |= LANDWALK;
	if (strustr (card.Spec, "BAND")) card.Abilities |= BANDING;
	if (strustr (card.Spec, "FLANK")) card.Abilities |= FLANKING;
	if (strustr (card.Spec, "STRIKE")) card.Abilities |= FIRSTSTRIKE;
	if (strustr (card.Spec, "TRAMPLE")) card.Abilities |= TRAMPLE;
	if (strustr (card.Spec, "RAMPAGE")) card.Abilities |= RAMPAGE;

	if (!memicmp (card.Type, "SUM", 3)) card.Abilities |= SUMMON;
	if (!memicmp (card.Type, "ENC", 3)) card.Abilities |= ENCHANTMENT;
	if (!memicmp (card.Type, "SOR", 3)) card.Abilities |= SORCERY;
	if (!memicmp (card.Type, "INS", 3)) card.Abilities |= INSTANT;
	if (!memicmp (card.Type, "INT", 3)) card.Abilities |= INTERRUPT;
	if (!memicmp (card.Type, "MAN", 3)) card.Abilities |= MANASOURCE;
	if (!memicmp (card.Type, "ART", 3)) card.Abilities |= ARTIFACT;
	BOOL other = !(card.Abilities & (SUMMON | ENCHANTMENT | SORCERY | INSTANT | INTERRUPT | MANASOURCE | ARTIFACT));
	if (other && strustr (card.Type, "LAND")) card.Abilities |= LAND;

	if ((card.Abilities & SUMMON) || ((card.Abilities & ARTIFACT) && strustr (card.Type, "CREATURE")))
        card.Abilities |= CREATURE;
    if (card.Abilities & (SUMMON | ENCHANTMENT | ARTIFACT | LAND))
        card.Abilities |= PERMANENT;
    if (card.Abilities & (INSTANT | INTERRUPT | MANASOURCE))
        card.Abilities |= FASTEFFECT;

    if (strchr (card.Cast, 'R'))
        card.Abilities |= RED;
    if (strchr (card.Cast, 'G'))
        card.Abilities |= GREEN;
    if (strchr (card.Cast, 'W'))
        card.Abilities |= WHITE;
    if (strchr (card.Cast, 'B'))
        card.Abilities |= BLACK;
    if (strchr (card.Cast, 'U'))
        card.Abilities |= BLUE;

    if (strchr (card.Cast, '0') || strchr (card.Cast, '1') || strchr (card.Cast, '2') ||
        strchr (card.Cast, '3') || strchr (card.Cast, '4') || strchr (card.Cast, '5') ||
        strchr (card.Cast, '6') || strchr (card.Cast, '7') || strchr (card.Cast, '8') ||
        strchr (card.Cast, '9') || strchr (card.Cast, 'X'))
        card.Abilities |= COLORLESS;

    int colors = card.Abilities & (RED | GREEN | WHITE | BLACK | BLUE);
    if (colors && (colors != RED) && (colors != GREEN) && (colors != WHITE) && (colors != BLACK) && (colors != BLUE))
        card.Abilities |= MULTICOLOR;

    card.Edition = 0;

    char *ptr = card.Freq;
    char freq [10];
    char c;

    while (*ptr) {
        c = *ptr++;
        GetFreq (freq, ptr);
        switch (c) {
            case 'g':
                if (strcmp (freq, "-"))
                    card.Edition |= ED_UL;
                GetFreq (freq, ptr);
                if (strcmp (freq, "-"))
                    card.Edition |= ED_RV;
                GetFreq (freq, ptr);
                if (strcmp (freq, "-"))
                    card.Edition |= ED_4E;
                GetFreq (freq, ptr);
                if (strcmp (freq, "-"))
                    card.Edition |= ED_5E;
                break;
            case 'a':
                card.Edition |= ED_AI;
                break;
            case 'n':
                card.Edition |= ED_AN;
                break;
            case 'q':
                card.Edition |= ED_AQ;
                break;
            case 'c':
                card.Edition |= ED_CH;
                break;
            case 'd':
                card.Edition |= ED_DK;
                break;
            case 'f':
                card.Edition |= ED_FE;
                break;
            case 'h':
                card.Edition |= ED_HL;
                break;
            case 'i':
                card.Edition |= ED_IA;
                break;
            case 'l':
                card.Edition |= ED_LG;
                break;
            case 'm':
                card.Edition |= ED_MI;
                break;
            case 'w':
                card.Edition |= ED_WL;
                break;
            case 't':
                card.Edition |= ED_TE;
                break;
            case 'v':
                card.Edition |= ED_VI;
                break;
            case 'p':
                card.Edition |= ED_P;
                break;
        }
    }
}

static istream& operator>> (istream& stream, CARD_INFO& card)
{
	stream >> card.Key;
	SkipToDelim (stream);
	ReadToDelim (stream, card.Name);
	ReadToDelim (stream, card.Type);
	ReadToDelim (stream, card.Cast);
	ReadToDelim (stream, card.Power);
	ReadToDelim (stream, card.Tough);
	ReadToDelim (stream, card.Spec);
	ReadToDelim (stream, card.Desc);
	ReadToDelim (stream, card.Freq);
	SetAbilities (card);

	return stream;
}

static ostream& operator<< (ostream& stream, CARD_INFO& card)
{
	stream << card.Key << ' ' << DELIM;
	stream << card.Name << DELIM;
	stream << card.Type << DELIM;
	stream << card.Cast << DELIM;
	stream << card.Power << DELIM;
	stream << card.Tough << DELIM;
	stream << card.Spec << DELIM;
	stream << card.Desc << DELIM;
	stream << card.Freq << DELIM;
	stream << endl;
	return stream;
}

static void SearchAppendFind (char *title, const char *find, BOOL& connect)
{
    static char Last [256];

    if (*find && (!connect || strcmp (find, Last))) {
        if (connect) strcat (title, " And ");
        strcpy (Last, find);
        strcat (title, find);
        connect = TRUE;
    }
}

const char * SearchTitle (const CARD_SEARCH *search)
{
    const int MAXTITLE = 5000;  // not sure how big this should be ...
    static epp_Heap buffer;
    char *title;
    BOOL connect;

    if (!buffer.Address ())
        buffer.Alloc (MAXTITLE);

    title = (char *) buffer.Address ();
    *title = NULL;

    connect = FALSE;

    SearchAppendFind (title, search->Name, connect);
    SearchAppendFind (title, search->Type, connect);
    SearchAppendFind (title, search->Spec, connect);
    SearchAppendFind (title, search->Desc, connect);
    SearchAppendFind (title, ((search->SetFlags | search->ClearFlags) & FLAGS_COLORS)? "Color" : "", connect);
    SearchAppendFind (title, ((search->SetFlags | search->ClearFlags) & FLAGS_TYPE)? "Type" : "", connect);
    SearchAppendFind (title, ((search->SetFlags | search->ClearFlags) & FLAGS_ABILITIES)? "Abilities" : "", connect);
    SearchAppendFind (title, ((search->SetFlags | search->ClearFlags) & FLAGS_ABILITIES)? "Abilities" : "", connect);
    SearchAppendFind (title, (search->SetEdition | search->ClearEdition)? "Set" : "", connect);

    if (!*title)
        LoadString (IDS_ALLCARDS, title);

    return title;
}

struct SEARCH_DATA
{
    CardList *Source;
    CardList *Target;
    CARD_SEARCH Search;
    BOOL AllCards;
};

static DWORD pascal _SearchLibrary (void *param)
{
    SEARCH_DATA *data = (SEARCH_DATA *) param;
    epp_ListBox source;
    epp_ListBox target;

    target.Attach (data->Target->GetCardList ());

    if (data->Source) {
        source.Attach (data->Source->GetCardList ());
        Library.FindCards (source, target, (data->AllCards)? NULL : &data->Search);
    } else
        Library.FindCards (target, (data->AllCards)? NULL : &data->Search);

    if (!target.GetCount ())
        Error ("No matching cards found", MB_ICONEXCLAMATION);

    data->Target->UpdateStats ();
    BusyEnd ();

    delete data;
    return 0;
}

void SearchLibrary (CardList *source, CardList *target, const CARD_SEARCH *search)
{
    epp_Thread thread;
    SEARCH_DATA *data;

    data = new SEARCH_DATA;
    data->Source = source;
    data->Target = target;

    if (search) {
        data->Search = *search;
        data->AllCards = FALSE;
    } else
        data->AllCards = TRUE;

    thread.Create (_SearchLibrary, data);
    BusyBegin (IDS_SEARCHING, *target);
}