////////////////////
// Filter Classes //
////////////////////

#include "Filter.h"
#include "Basic IO.h"
#include <iomanip.h>
#include <stdlib.h>

const int MAXLINE = 1000;

static void Trim (char *source)
{
    char *ptr;
    int end;

    end = strlen (source);

    if (end) {
        ptr = source + end - 1;
        while (end-- && (*ptr <= ' '))
            *ptr-- = NULL;

        ptr = source;
        while (*source && *source <= ' ')
            source ++;

        if (source != ptr) {
            while (*source)
                *ptr++ = *source++;
            *ptr = NULL;
        }
    }
}

static BOOL ValidateType (const char *type)
{
    return !memicmp (type, "SUMMON", 6) ||
        !memicmp (type, "ENCHANT", 7) ||
        !memicmp (type, "ARTIFACT", 8) ||
        !memicmp (type, "SORCERY", 7) ||
        !memicmp (type, "INSTANT", 7) ||
        !memicmp (type, "INTERRUPT", 9) ||
        !memicmp (type, "MANA", 4) ||
        !memicmp (type, "LAND", 4) ||
        !memicmp (type, "LEGENDARY", 9);
}

static BOOL IsCreature (const char *type)
{
    if (!memicmp (type, "SUMMON", 6))
        return TRUE;
    if (!memicmp (type, "ARTIFACT", 8) && strustr (type, "CREATURE"))
        return TRUE;
    return FALSE;
}

Line::Line ()
{
    _Line = new char [MAXLINE];
    _Loaded = FALSE;
}

Line::~Line ()
{
    delete _Line;
}

BOOL Line::Load (istream& stream)
{
    char *ptr;

    _Index = 0;

    if (!_Loaded) {
        while (TRUE) {
    		stream.getline (_Line, MAXLINE);
    		if (!stream)
    		    return FALSE;
    		if (strlen (_Line))
    		    break;
    	}

        _Indent = 0;
        ptr = _Line;
        while (*ptr++ == ' ')
            _Indent ++;
    }
    _Loaded = TRUE;
    return TRUE;
}

void Line::Finished ()
{
    _Loaded = FALSE;
}

void Line::SkipSpace ()
{
    while (_Line [_Index] && (_Line [_Index] <= ' '))
        _Index ++;
}

void Line::SkipColumns (int cols)
{
    while (cols-- && _Line [_Index])
        _Index ++;
}

void Line::ReadColumns (char *target, int cols)
{
    char *ptr = target;

    while (cols-- && _Line [_Index])
        *ptr++ = _Line [_Index++];
    *ptr = NULL;

    Trim (target);
}

void Line::ReadToChar (char *target, char stop)
{
    char *ptr = target;

    while ((_Line [_Index] != stop) && _Line [_Index])
        *ptr++ = _Line [_Index++];
    *ptr = NULL;

    if (_Line [_Index]) _Index ++;

    Trim (target);
}

void Line::ReadToEnd (char *target)
{
    char *ptr = target;

    while (_Line [_Index])
        *ptr++ = _Line [_Index++];
    *ptr = NULL;

    Trim (target);
}

DevjoeFilter::DevjoeFilter ()
{
    _Desc = new char [2000];
}

DevjoeFilter::~DevjoeFilter ()
{
    delete _Desc;
}

BOOL DevjoeFilter::Read (istream& stream, CARD_INFO& card)
{
    BOOL valid;
    int indent;

    valid = FALSE;

    while (!valid) {
        memset (&card, 0, sizeof (card));

        if (!_Line.Load (stream))
            return FALSE;
        if (_Line.Indent ())
            _Line.Finished ();
        else {
            _Line.SkipColumns (7);
            _Line.ReadColumns (card.Name, 33);
            _Line.ReadToChar (card.Freq, ' ');
            _Line.SkipSpace ();
            _Line.ReadToEnd (card.Cast);
            _Line.Finished ();
            if (!_Line.Load (stream))
                return FALSE;
            indent = _Line.Indent ();
            if (indent > 0) {
                _Line.ReadColumns (card.Type, 29);
                valid = ValidateType (card.Type);
            }
        }
    }

    if (IsCreature (card.Type)) {
        _Line.ReadToChar (card.Power, '/');
        _Line.ReadToChar (card.Tough, ',');
    }
    _Line.ReadToEnd (card.Spec);
    _Line.Finished ();

    *_Desc = NULL;

    while (_Line.Load (stream) && (_Line.Indent () > indent)) {
        _Line.ReadToEnd (_Desc);
        if (*card.Desc)
            strcat (card.Desc, " ");
        strcat (card.Desc, _Desc);
        _Line.Finished ();
    }

    return TRUE;
}

void ListFilter::ReadHeader (istream& stream, char *title)
{
    _Line.Load (stream);
    _Line.ReadToEnd (title);
    _Line.Finished ();
}

BOOL ListFilter::Read (istream& stream, int& count, char *name)
{
    char temp [256];

    while (TRUE) {
        if (!_Line.Load (stream))
            return FALSE;
        _Line.SkipSpace ();
        _Line.ReadToChar (temp, ' ');
        _Line.ReadToEnd (name);
        _Line.Finished ();
        count = atoi (temp);
        if (_Line.Indent () && count)
            break;
    }
    return TRUE;
}

void ListFilter::WriteHeader (ostream& stream, const char *title)
{
    stream << title << endl;
}

void ListFilter::Write (ostream& stream, int count, const char *name)
{
    stream << ' ' << setw (3) << count << ' ' << name << endl;
}